-- Pull in the wezterm API, some of its modules, and plugins
local wezterm      = require 'wezterm'
local act          = wezterm.action
local merge        = require 'merge'
local mux          = wezterm.mux
local resurrect    = require 'resurrect-config'
local smart_splits = require 'smart-splits-setup'

-- --------------------------------------------------------------------
-- CONFIGURATION
-- --------------------------------------------------------------------

-- This table will hold the configuration.
local config = {}

-- In newer versions of wezterm, use the config_builder which will
-- help provide clearer error messages
if wezterm.config_builder then
  config = wezterm.config_builder()
end

config.adjust_window_size_when_changing_font_size = false
config.automatically_reload_config = true
config.color_scheme = 'Solarized (dark) (terminal.sexy)'
config.enable_scroll_bar = true
config.enable_wayland = true
-- config.font = wezterm.font('Hack')
config.font = wezterm.font('MonaspiceNe NFP')
config.font_size = 12.0
config.hide_tab_bar_if_only_one_tab = false
-- The leader is similar to how tmux defines a set of keys to hit in order to
-- invoke tmux bindings. Binding to ctrl-a here to mimic tmux
config.leader = { key = 'a', mods = 'CTRL', timeout_milliseconds = 2000 }
config.mouse_bindings = {
    -- Open URLs with Ctrl+Click
    {
        event = { Up = { streak = 1, button = 'Left' } },
        mods = 'CTRL',
        action = act.OpenLinkAtMouseCursor,
    }
}
config.pane_focus_follows_mouse = true
config.scrollback_lines = 5000
config.tiling_desktop_environments = {
    'Wayland',
}
config.use_dead_keys = false
config.warn_about_missing_glyphs = false
config.window_decorations = "TITLE | RESIZE"
config.window_padding = {
    left = 0,
    right = 0,
    top = 0,
    bottom = 0,
}

-- Tab bar
config.use_fancy_tab_bar = true
config.tab_bar_at_bottom = true
config.switch_to_last_active_tab_when_closing_tab = true
config.tab_max_width = 32
config.colors = {
    quick_select_label_bg = { Color = '#fdf6e3' },
    quick_select_label_fg = { Color = '#073642' },
    tab_bar = {
        active_tab = {
            fg_color = '#073642',
            bg_color = '#2aa198',
        }
    }
}

-- Add items to launch menu
config.launch_menu = {
    {
        label = 'mwop',
        cwd = wezterm.home_dir .. '/git/weierophinney/mwop.net',
    },
    {
        label = 'onedrive',
        cwd = wezterm.home_dir .. '/OneDrive',
    },
    {
        label = 'top',
        args = { 'top' },
    },
}

-- Custom key bindings
config.keys = {
    -- Show the launcher
    {
        key = 'm',
        mods = 'LEADER',
        action = act.ShowLauncher,
    },

    -- Copy mode
    {
        key = '[',
        mods = 'LEADER',
        action = act.ActivateCopyMode,
    },

    -- ----------------------------------------------------------------
    -- TABS
    --
    -- Where possible, I'm using the same combinations as I would in tmux
    -- ----------------------------------------------------------------

    -- Show tab navigator; similar to listing panes in tmux
    {
        key = 'w',
        mods = 'LEADER',
        action = act.ShowTabNavigator,
    },

    -- Create a tab (alternative to Ctrl-Shift-Tab)
    {
        key = 'c',
        mods = 'LEADER',
        action = act.SpawnTab 'CurrentPaneDomain',
    },

    -- Rename current tab; analagous to command in tmux
    {
        key = ',',
        mods = 'LEADER',
        action = act.PromptInputLine {
            description = 'Enter new name for tab',
            action = wezterm.action_callback(
                function(window, pane, line) -- luacheck: ignore 212
                    if line then
                        window:active_tab():set_title(line)
                    end
                end
            ),
        },
    },

    -- Move to next/previous TAB
    {
        key = 'n',
        mods = 'LEADER',
        action = act.ActivateTabRelative(1),
    },
    {
        key = 'p',
        mods = 'LEADER',
        action = act.ActivateTabRelative(-1),
    },

    -- Close tab
    {
        key = '&',
        mods = 'LEADER|SHIFT',
        action = act.CloseCurrentTab{ confirm = true },
    },

    -- ----------------------------------------------------------------
    -- PANES
    --
    -- These are great and get me most of the way to replacing tmux
    -- entirely, particularly as you can use "wezterm ssh" to ssh to another
    -- server, and still retain Wezterm as your terminal there.
    --
    -- Note that these only define creating splits, relative motion
    -- (next/previous), zooming, swapping, and killing panes; actual directional
    -- motions between panes or resizing them are handled by smart splits.
    -- ----------------------------------------------------------------

    -- Vertical split
    {
        -- |
        key = '|',
        mods = 'LEADER|SHIFT',
        action = act.SplitPane {
            direction = 'Right',
            size = { Percent = 50 },
        },
    },

    -- Horizontal split
    {
        -- -
        key = '-',
        mods = 'LEADER',
        action = act.SplitPane {
            direction = 'Down',
            size = { Percent = 50 },
        },
    },

    -- Close/kill active pane
    {
        key = 'x',
        mods = 'LEADER',
        action = act.CloseCurrentPane { confirm = true },
    },

    -- Swap active pane with another one
    {
        key = '{',
        mods = 'LEADER|SHIFT',
        action = act.PaneSelect { mode = "SwapWithActiveKeepFocus" },
    },

    -- Zoom current pane (toggle)
    {
        key = 'z',
        mods = 'LEADER',
        action = act.TogglePaneZoomState,
    },
    {
        key = 'f',
        mods = 'ALT',
        action = act.TogglePaneZoomState,
    },

    -- Move to next/previous pane
    {
        key = ';',
        mods = 'LEADER',
        action = act.ActivatePaneDirection('Prev'),
    },
    {
        key = 'o',
        mods = 'LEADER',
        action = act.ActivatePaneDirection('Next'),
    },

    -- ----------------------------------------------------------------
    -- Workspaces
    --
    -- These are roughly equivalent to tmux sessions.
    -- ----------------------------------------------------------------

    -- Attach to muxer
    {
        key = 'a',
        mods = 'LEADER',
        action = act.AttachDomain 'unix',
    },

    -- Detach from muxer
    {
        key = 'd',
        mods = 'LEADER',
        action = act.DetachDomain { DomainName = 'unix' },
    },

    -- Show list of workspaces
    {
        key = 's',
        mods = 'LEADER',
        action = act.ShowLauncherArgs { flags = 'WORKSPACES' },
    },

    -- Rename current session; analagous to command in tmux
    {
        key = '$',
        mods = 'LEADER|SHIFT',
        action = act.PromptInputLine {
            description = 'Enter new name for session',
            action = wezterm.action_callback(
                function(window, pane, line) -- luacheck: ignore 212
                    if line then
                        mux.rename_workspace(
                            window:mux_window():get_workspace(),
                            line
                        )
                    end
                end
            ),
        },
    },
}

-- --------------------------------------------------------------------
-- Smart splits
--
-- See https://github.com/mrjones2014/smart-splits.nvim
--
-- Allows moving and resizing panes easily, as well as navigation between
-- wezterm and nvim panes
-- --------------------------------------------------------------------
config.keys = merge.all(config.keys, smart_splits.keys)

-- --------------------------------------------------------------------
-- resurrect.wezterm
--
-- See https://github.com/MLFlexer/resurrect.wezterm
-- See resurrect.lua
-- --------------------------------------------------------------------
config.keys = merge.all(config.keys, resurrect.keys)

-- Powerline for tab bar
require 'powerline'

-- Tab status
require 'tab-status'

-- Plugin management
-- Automatically update plugins
-- wezterm.plugin.update_all()

-- and finally, return the configuration to wezterm
return config